/**
 * \file: caam.h
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: Secure Data Container
 * \brief : defines, enums specific to imx6
 *
 * <b>SDC private header</b>
 *
 * \author: Ian Molton (ian.molton@codethink.co.uk)
 *	   Norbert Uetrecht (nuetrecht@de.adit-jv.com)
 *	   Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2014 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#ifndef __SDC_LIB_CAAM_H_
#define __SDC_LIB_CAAM_H_

#include <stdint.h>
#include <stdbool.h>
#include <sdc.h>
#include <private/sdc_arch.h>
#include <sdc_session.h>
#include <sdc_op_common.h>
#include <linux/caam_ee.h>

#define CAAM_DEVICE ("/dev/" CAAM_EE_DEV_NAME)
#define AES_CCM_TOTAL_HEADER_BYTES 15

/* key id */
typedef uint32_t caam_key_id;

/** Get libsdc error value for open errno
 *
 * \param err         ioctl error number
 * \return libsdc error
 */
sdc_error_t error_from_open_errno(int err);

/** Get libsdc error value for ioctl errno
 *
 * \param err         ioctl error number
 * \param ioctl_cmd   the ioctl
 * \param ioctl_flags flags of the ioctl
 * \return libsdc error
 */
sdc_error_t error_from_ioctl_errno(int err, unsigned long int ioctl_cmd, uint32_t ioctl_flags);

typedef enum {
    WRAP,
    UNWRAP
} caam_wrap_unwrap_op;

/**
 * \brief IMX6 specific sdc_wrap_unwrap_type struct
 */
struct sdc_wrap_unwrap_type {
    sdc_wrap_unwrap_alg_t alg;
    sdc_wrap_unwrap_blk_t blk;
    uint32_t arch_alg_flag;
    uint32_t arch_alg_idx;
    uint32_t arch_blk_flag;
    uint32_t arch_blk_idx;
};

typedef enum {
    ENCRYPT,
    DECRYPT
} caam_encrypt_decrypt_op;

/**
 * \brief IMX6 specific sdc_encrypt_decrypt_type struct
 */
struct sdc_encrypt_decrypt_type {
    sdc_encrypt_decrypt_alg_t alg;
    sdc_encrypt_decrypt_blk_t blk;
    bool is_symmetric;
    uint32_t arch_alg_flag; /* only needed for symmetric */
    uint32_t arch_alg_idx;  /* only needed for symmetric */
    uint32_t arch_blk_flag; /* only needed for symmetric */
    uint32_t arch_blk_idx;  /* only needed for symmetric */
};

typedef enum {
    SIGN,
    VERIFY
} caam_sign_verify_op;

/**
 * \brief IMX6 specific sdc_sign_verify_type struct
 */
struct sdc_sign_verify_type {
    sdc_sign_verify_alg_t alg;
    sdc_sign_verify_hash_t hash;
    uint64_t opt_bmsk;
    uint32_t arch_alg_flag;
    uint32_t arch_alg_idx;
    uint32_t arch_hash_flag;
    uint32_t arch_hash_idx;
};

/**
 * \brief IMX6 specific sdc_sign_verify_type struct
 */
struct sdc_dgst_type {
    sdc_dgst_hash_t hash;
    uint32_t arch_hash_flag;
    uint32_t arch_hash_idx;
};

SDC_ARCH_API void caam_keylen_bmsk_to_sdc(uint32_t caam_bmsk, sdc_key_len_bmsk_t *bmsk, sdc_key_len_t *max_len);

SDC_ARCH_API sdc_error_t caam_check_session_key_set(sdc_session_t *session);

SDC_ARCH_API int caam_perform_encrypt_decrypt_ioctl_operation(sdc_session_t *session, struct caam_ee_encrypt_decrypt_params *iodata,
                                                              const uint8_t *in_data_ptr, uint8_t *out_data_ptr,
                                                              size_t in_len, size_t out_len, uint32_t iodata_flags);

SDC_ARCH_API int caam_perform_sign_verify_ioctl_operation(sdc_session_t *session, struct caam_ee_sign_verify_params *iodata,
                                                          const uint8_t *in_data_ptr, size_t in_len,
                                                          uint8_t *tag_out_data, size_t tag_out_data_len,
                                                          const uint8_t *tag_in_data, size_t tag_in_data_len,
                                                          uint32_t iodata_flags);

SDC_ARCH_API int caam_perform_dgst_ioctl_operation(sdc_session_t *session, struct caam_ee_dgst_params *iodata,
                                                   const uint8_t *in_data_ptr, size_t in_len,
                                                   uint8_t *dgst_out_data, size_t dgst_out_data_len,
                                                   uint32_t iodata_flags);

SDC_ARCH_API int caam_perform_wrap_unwrap_ioctl_operation(sdc_session_t *session, struct caam_ee_wrap_unwrap_params *iodata,
                                                          const uint8_t *in_data, const size_t in_data_len,
                                                          uint8_t *out_tag_data, size_t out_tag_data_len,
                                                          const uint8_t *in_tag_data, size_t in_tag_data_len,
                                                          uint8_t *out_data_ptr, size_t out_data_len,
                                                          uint32_t iodata_flags);

SDC_ARCH_API sdc_error_t caam_perform_abort_operation(sdc_session_t *session);

SDC_ARCH_API bool caam_is_op_in_progress(sdc_session_t *session);

#endif
